/**
 * \file: grl_bmpdec_png.c
 *
 * version: $Id: grl_bmpdec_png.c,v 1.14 2010/02/10 14:39:25 tkniep Exp $
 *
 * Implementation of the wrapper between the PNG lib and the
 * SVG Bitmap Decoder
 *
 * \component: SVG Bitmap Decoder (SVGBMPDEC)
 *
 * \author: T. Kniep (tkniep@de.adit-jv.com)
 *
 * \copyright: (c) 2009 ADIT Corporation
 *
 ***********************************************************************/

#include <string.h>
#include <stdlib.h>

#include <svg_bitmap_decoder.h>
#include <grl_bitmap_decoder_util.h>
#include "grl_bmpdec_png.h"
#include "png.h"

/* new API for libpng 1.6 */
#if PNG_LIBPNG_VER_MAJOR == 1 && PNG_LIBPNG_VER_MINOR >= 6
	#include <setjmp.h>
	#define PNG_SETJMP_SUPPORTED
	#include <pngstruct.h>
#endif

/*******************************************************************************************
 *   Internal function prototypes
 *******************************************************************************************/


/**
 * Read data function.
 * Callback function used by PNG lib to read compressed PNG data.
 *
 * \param[in]   png_ptr                 Pointer to data info structure
 * \param[in]   data                    Pointer to where data will be read to
 * \param[in]   length                  Length of data
 *
 */
static   void      GRL_BMPDEC_png_read_func ( png_structp        png_ptr,
                                             png_bytep          data,
                                             png_size_t         length );



/**
 * Advanced read data function for palette replacement.
 * Callback function used by PNG lib to read compressed PNG data. In addition
 * to what GRL_BMPDEC_png_read_func does, this function also replaces
 * the PLTE and rRNS chunks in order to implement the palette replacement feature.
 *
 * \param[in]   png_ptr                 Pointer to data info structure
 * \param[in]   data                    Pointer to where data will be read to
 * \param[in]   length                  Length of data
 *
 */
static   void      GRL_BMPDEC_png_read_func_advanced ( png_structp png_ptr,
                                                      png_bytep         data,
                                                      png_size_t        length );


/**
 * Check and prepare external palettes.
 * The validity of application-provided palettes is checked and
 * the structures are prepared for replacement.
 *
 * \param[in]   p_image                 Pointer to image definition structure
 * \param[out]  p_plte_data             Pointer to palette data structure
 *
 * \return      SVG_NO_ERROR            Palettes have been prepared successfully
 * \return      SVG_INVALID_VALUE       Chunk is not valid
 */
static SVGError GRL_BMPDEC_palette_replace( SVGImage                   *p_image,
                                           GRL_BMPDEC_plte_data       *p_plte_data );


/**
 * Set the decoder configuration based on the selected destination type.
 *
 * \param[in]   p_image                 Pointer to image definition structure
 * \param[out]  dec_conf                Pointer to updated decoder configuration
 *
 * \return      SVG_NO_ERROR            Destination was set successfully
 * \return      SVG_INVALID_OPERATION   The destination mode is invalid
 * \return      SVG_INVALID_VALUE       The image data pointer is NULL
 */
static GRL_error GRL_BMPDEC_png_dest_set( SVGImage *p_image,
                                         GRL_BMPDEC_png_dec_conf *dec_conf );


/**
 * Set the color conversion transformations of the pnglib based on the 
 * bit depth and color type of the image.
 *
 * \param[in]   png_ptr                 Pointer to data info structure
 * \param[in]   bit_depth               Bit depth of image
 * \param[in]   color_type              Color type of image
 *
 */
static void GRL_BMPDEC_png_color_set( png_structp png_ptr,
                                     SVGInt32 bit_depth,
                                     SVGInt32 color_type );


/*******************************************************************************************
 *   Function implementations
 *******************************************************************************************/

static void GRL_BMPDEC_png_read_func_advanced (
    png_structp         png_ptr,
    png_bytep           data,
    png_size_t          length
    )
{
    void        *p_work_vd;
    U8          *p_work_u8;
    GRL_BMPDEC_plte_data *info = NULL;
    SVGBoolean foreward = SVG_TRUE;
    SVGBoolean do_copy  = SVG_TRUE;

    info = (GRL_BMPDEC_plte_data*)png_ptr->io_ptr;
    p_work_vd = info->io_ptr;
    if(png_ptr->color_type == PNG_COLOR_TYPE_PALETTE)
    {
        p_work_u8 = (U8 *)p_work_vd;
        /* do the loop for the external palette */
        if(SVG_TRUE == info->external_PLTE_in_use)
        {
            if(length == GRL_BMPDEC_PNG_PLTE_CHANNELS)
            {
                /* copy the entries of the external chunk */
                memcpy( data,
                        &info->external_PLTE[GRL_BMPDEC_PNG_PLTE_CHANNELS * info->external_PLTE_entry],
                        length );
                info->external_PLTE_entry++;
                /* skip the entries of the original buffer for last entry */
                if((GRL_BMPDEC_PNG_PLTE_CHANNELS * info->external_PLTE_entry) > info->internal_PLTE_length)
                {
                    foreward = SVG_FALSE;
                }
                do_copy = SVG_FALSE;
            }
            else
            {
                /* now the CRC has to be copied */
                memcpy( data, &info->external_PLTE_CRC, length );
                /* the chunk is finished */
                info->external_PLTE_in_use = SVG_FALSE;
                do_copy = SVG_FALSE;
                /* when external palette has less entries then internal then
                   skip the remaining entries */
                if((info->external_PLTE_entry * GRL_BMPDEC_PNG_PLTE_CHANNELS) < info->internal_PLTE_length)
                {
                    length += (info->internal_PLTE_length -
                               (info->external_PLTE_entry * GRL_BMPDEC_PNG_PLTE_CHANNELS));
                }
            }
        }
        else
        {
            /* copy the length of the external palette when next entry is PLTE chunk name  */
            if((!memcmp(GRL_BMPDEC_PNG_PLTE_CHUNK, (signed char*)(void*)&p_work_u8[GRL_BMPDEC_PNG_WORD_SIZE], length)) &&
               (NULL != info->external_PLTE))
            {
                memcpy( data, &info->external_PLTE_length[0], length );
                /* remember the original chunk length */
                info->internal_PLTE_length = png_get_uint_32(p_work_u8);
                do_copy = SVG_FALSE;
            }

            /* copy the chunk name */
            if((!memcmp(GRL_BMPDEC_PNG_PLTE_CHUNK, (signed char*)(void*)p_work_u8, length)) &&
               (NULL != info->external_PLTE))
            {
                memcpy( data, GRL_BMPDEC_PNG_PLTE_CHUNK, length );
                info->external_PLTE_entry = 0;
                info->external_PLTE_in_use = SVG_TRUE;
                do_copy = SVG_FALSE;
            }
        }

        /* do the loop for the external trasparency */
        if(SVG_TRUE == info->external_TRNS_in_use)
        {
            if(length != GRL_BMPDEC_PNG_WORD_SIZE)
            {
                /* copy all the entries of the external chunk */
                memcpy( data, &info->external_TRNS[0], length );
                /* skip all entries within the original transparency palette */
                length = info->internal_TRNS_length;
                do_copy = SVG_FALSE;
            }
            else
            {
                /* now the CRC has to be copied */
                memcpy( data, &info->external_TRNS_CRC, length );
                /* the chunk is finished */
                info->external_TRNS_in_use = SVG_FALSE;
                do_copy = SVG_FALSE;
            }
        }
        else
        {
            /* copy the length of the external palette */
            if((!memcmp(GRL_BMPDEC_PNG_TRNS_CHUNK, (signed char*)(void*)&p_work_u8[GRL_BMPDEC_PNG_WORD_SIZE], length)) &&
               (NULL != info->external_TRNS))
            {
                memcpy( data, &info->external_TRNS_length[0], length );
                /* remember the original chunk length */
                info->internal_TRNS_length = png_get_uint_32(p_work_u8);
                do_copy = SVG_FALSE;
            }

            /* copy the chunk name */
            if((!memcmp(GRL_BMPDEC_PNG_TRNS_CHUNK, (signed char*)(void*)p_work_u8, length)) &&
               (NULL != info->external_TRNS))
            {
                memcpy( data, &info->external_TRNS_header[0], length );
                info->external_TRNS_in_use = SVG_TRUE;
                do_copy = SVG_FALSE;
            }
        }

        if(SVG_TRUE == do_copy)
        {
            memcpy( data, p_work_u8, length );
        }

        if(SVG_TRUE == foreward)
        {
            p_work_u8 = &p_work_u8[length];
            p_work_vd = (void *)p_work_u8;
            info->io_ptr = p_work_vd;
        }
    }
    else
    {
        memcpy( data, p_work_vd, length );
        p_work_u8 = (U8 *)p_work_vd;
        p_work_u8 = &p_work_u8[length];
        p_work_vd = (void *)p_work_u8;
        info->io_ptr = p_work_vd;
    }

}


static void GRL_BMPDEC_png_read_func ( png_structp png_ptr,
                                png_bytep   data,
                                png_size_t  length )
{
        void    *p_work_vd;
        U8      *p_work_u8;

        memcpy( data, png_ptr->io_ptr, length );
        p_work_vd = (void *)png_ptr->io_ptr;
        p_work_u8 = (U8 *)p_work_vd;
        p_work_u8 = &p_work_u8[length];
        p_work_vd = (void *)p_work_u8;
        png_ptr->io_ptr = p_work_vd;
}


static SVGError GRL_BMPDEC_palette_replace( SVGImage                   *p_image,
                                     GRL_BMPDEC_plte_data       *p_plte_data )
{
    SVGError    ret_err         = SVG_NO_ERROR;
    SVGUint32   crc             = 0;

    SVG_BMP_U("ENTER INTO GRL_BMPDEC_PALETTE_REPLACE");

    /* Handle PLTE chunk (color palette) */
    if (p_image->attribute.png.plte_chunk_size > 0)
    {
        p_plte_data->use_external_palette = SVG_TRUE;

        /* Check if chunk name is contained in external chunk -> skip it */
        if( !memcmp( GRL_BMPDEC_PNG_PLTE_CHUNK, (signed char*)p_image->attribute.png.plte_chunk, GRL_BMPDEC_PNG_WORD_SIZE))
        {
            p_image->attribute.png.plte_chunk_size -= GRL_BMPDEC_PNG_WORD_SIZE;
            p_plte_data->external_PLTE        = (SVGUint8*)&p_image->attribute.png.plte_chunk[GRL_BMPDEC_PNG_WORD_SIZE];
        }
        else
        {
            ret_err = SVG_INVALID_VALUE;
            SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_PALETTE_REPLACE");
        }

        /* Check if chunk data contains CRC information */
        if ( (SVG_NO_ERROR == ret_err) &&
             (0 != (p_image->attribute.png.plte_chunk_size % GRL_BMPDEC_PNG_PLTE_CHANNELS)) )
        {
            p_image->attribute.png.plte_chunk_size -= GRL_BMPDEC_PNG_WORD_SIZE;
            /* now only palette entries should be there, when not leave... */
            if(0 != (p_image->attribute.png.plte_chunk_size % GRL_BMPDEC_PNG_PLTE_CHANNELS))
            {
                ret_err = SVG_INVALID_VALUE;
                SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_PALETTE_REPLACE");
            }
        }
        else
        {
            ret_err = SVG_INVALID_VALUE;
            SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_PALETTE_REPLACE");
        }

        /* Number of palette entries exceeds maximum entries */
        if ( (SVG_NO_ERROR == ret_err) &&
             ((GRL_BMPDEC_PNG_PLTE_MAX_ENTRIES * GRL_BMPDEC_PNG_PLTE_CHANNELS) < 
              p_image->attribute.png.plte_chunk_size) )
        {
            ret_err = SVG_INVALID_VALUE;
            SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_PALETTE_REPLACE");
        }

        if ( SVG_NO_ERROR == ret_err )
        {
            /* Copy the PLTE length */
            png_save_uint_32( (unsigned char*)(void*)&(p_plte_data->external_PLTE_length),
                              (SVGUint16)(p_image->attribute.png.plte_chunk_size) );

            /* Calculate CRC with chunk data */
            memset( &(p_plte_data->external_PLTE_CRC[0]), 0x00, GRL_BMPDEC_PNG_WORD_SIZE );
            crc = crc32(0, Z_NULL, 0);
            crc = crc32(crc, (unsigned char*)GRL_BMPDEC_PNG_PLTE_CHUNK, GRL_BMPDEC_PNG_WORD_SIZE);
            crc = crc32(crc, p_plte_data->external_PLTE, (SVGUint16)p_image->attribute.png.plte_chunk_size);
            png_save_uint_32((unsigned char*)&(p_plte_data->external_PLTE_CRC[0]), crc);

            /* Check calculated CRC against chunk CRC*/
            if( memcmp( (unsigned char*)&(p_plte_data->external_PLTE_CRC[0]),
                        &(p_plte_data->external_PLTE[p_image->attribute.png.plte_chunk_size]),
                        GRL_BMPDEC_PNG_WORD_SIZE ) != 0)
            {
                ret_err = SVG_INVALID_VALUE;
                SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_PALETTE_REPLACE");
            }
        }

        if ( SVG_NO_ERROR == ret_err )
        {
            /* Empty palette */
            if(0 == p_image->attribute.png.plte_chunk_size)
            {
                p_plte_data->external_PLTE = NULL;
            }
        }

    }
    
    /* Handle tRNS chunk (alpha palette) */
    if( (SVG_NO_ERROR == ret_err) &&
        (p_image->attribute.png.trns_chunk_size > 0) )
    {

        p_plte_data->use_external_palette = SVG_TRUE;

        /* Check if chunk name is contained in external chunk -> skip it */
        if( !memcmp( GRL_BMPDEC_PNG_TRNS_CHUNK, (signed char*)p_image->attribute.png.trns_chunk, GRL_BMPDEC_PNG_WORD_SIZE) )
        {
            memcpy(&(p_plte_data->external_TRNS_header[0]),
                   &(p_image->attribute.png.trns_chunk[0]),
                   GRL_BMPDEC_PNG_CHUNK_HDR_SIZE);
            p_image->attribute.png.trns_chunk_size -= GRL_BMPDEC_PNG_WORD_SIZE;
            p_plte_data->external_TRNS             = (SVGUint8*)(void*)&p_image->attribute.png.trns_chunk[GRL_BMPDEC_PNG_WORD_SIZE];
        }
        else
        {
            ret_err = SVG_INVALID_VALUE;
            SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_PALETTE_REPLACE(tRNS chunk)");
        }

        if ( SVG_NO_ERROR == ret_err )
        {
            /* Substract CRC length */
            p_image->attribute.png.trns_chunk_size -= GRL_BMPDEC_PNG_WORD_SIZE;

            /* Number of palette entries exceeds maximum entries */
            if(GRL_BMPDEC_PNG_PLTE_MAX_ENTRIES  < p_image->attribute.png.trns_chunk_size)
            {
                ret_err = SVG_INVALID_VALUE;
                SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_PALETTE_REPLACE(tRNS chunk)");
            }
        }

        if ( SVG_NO_ERROR == ret_err )
        {
            /* Copy chunk length */
            png_save_uint_32( (unsigned char*)(void*)&(p_plte_data->external_TRNS_length),
                              (SVGUint16)p_image->attribute.png.trns_chunk_size);

            /* tRNS length has not to be bigger then PLTE length */
            if( (NULL != p_image->attribute.png.plte_chunk) &&
                (0 != p_image->attribute.png.plte_chunk_size) &&
                ((GRL_BMPDEC_PNG_PLTE_CHANNELS * p_image->attribute.png.trns_chunk_size) 
                 > p_image->attribute.png.plte_chunk_size))
            {
                ret_err = SVG_INVALID_VALUE;
                SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_PALETTE_REPLACE(tRNS chunk)");
            }
        }

        if ( SVG_NO_ERROR == ret_err )
        {
            /* Calculate CRC with chunk data */
            memset( &(p_plte_data->external_TRNS_CRC[0]),
                    0x00,
                    GRL_BMPDEC_PNG_WORD_SIZE );
            crc = crc32(0, Z_NULL, 0);
            crc = crc32(crc, (unsigned char*)&(p_plte_data->external_TRNS_header[0]), GRL_BMPDEC_PNG_WORD_SIZE );
            crc = crc32(crc, p_plte_data->external_TRNS, (SVGUint16)p_image->attribute.png.trns_chunk_size);
            png_save_uint_32((unsigned char*)&(p_plte_data->external_TRNS_CRC[0]), crc);

            /* Check calculated CRC against chunk CRC*/
            if( (memcmp( (unsigned char*)&(p_plte_data->external_TRNS_CRC[0]),
                         &(p_plte_data->external_TRNS[p_image->attribute.png.trns_chunk_size]),
                         GRL_BMPDEC_PNG_WORD_SIZE ) != 0) &&
                (0x20 != (p_plte_data->external_TRNS_header[0] & 0x20)))
            {
                /* CRC is invalid, and mandatory due to TRNS chunk name */
                ret_err = SVG_INVALID_VALUE;
                SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_PALETTE_REPLACE(tRNS chunk)");
            }
        }

        if ( SVG_NO_ERROR == ret_err )
        {
            /* empty palette */
            if(0 == p_image->attribute.png.trns_chunk_size)
            {
                p_plte_data->external_TRNS = NULL;
            }
        }

    }
    
    SVG_BMP_U("EXIT FROM GRL_BMPDEC_PALETTE_REPLACE");

    return ret_err;
}


static GRL_error GRL_BMPDEC_png_dest_set( SVGImage *p_image, GRL_BMPDEC_png_dec_conf *dec_conf )
{
    SVGError    ret_err = SVG_NO_ERROR;

    /* Handle destination-specific settings */
    switch ( p_image->decode_mode )
    {
        case SVG_BMPDEC_MODE_MEMORY:
        {
            dec_conf->p_dst_mem  = p_image->dest.dest_mem.dest_mem;
            dec_conf->dst_width  = p_image->width;                        /* Assume width to be the width of the image */
            dec_conf->dst_mwidth = dec_conf->dst_width * GRL_BMPDEC_DEPTH_ARGB8888;
            dec_conf->dst_height = p_image->height;                       /* Assume height to be the height of the image */
            dec_conf->dst_xpos   = 0;                                     /* Position is always at beginning of mem block because */
            dec_conf->dst_ypos   = 0;                                     /* otherwise the size might not be sufficient */
            dec_conf->p_data     = (U8*)((void*)p_image->datap);
            dec_conf->src_clip   = p_image->clip;                                   
            break;
        }
        default:
        {
            /* Invalid destination mode */
            ret_err = SVG_INVALID_OPERATION;
            SVG_BMP_E("SVG_INVALID_OPERATION IN  GRL_BMPDEC_PNG_DEST_SET");
        }
    }

    if ( (SVG_NO_ERROR == ret_err) &&
         (dec_conf->p_data == NULL) )
    {
        ret_err = SVG_INVALID_VALUE;
        SVG_BMP_E("SVG_INVALID_OPERATION IN  GRL_BMPDEC_PNG_DEST_SET");
    }

    return ret_err;
}


static void GRL_BMPDEC_png_color_set( png_structp png_ptr, SVGInt32 bit_depth, SVGInt32 color_type )
{
    /*
     * If image is 16 bit (48bpp) color, strip the pixels down to 8 bit
     * because 48 bpp destination format is not supported.
     */
    if ( bit_depth == GRL_BMPDEC_PNG_BIT_DEPTH16 )
    {
        png_set_strip_16( png_ptr );
    }

    /*
     * If image bit depth is less than 8 bit, expand it to 8bpp.
     * because 1, 2, 4 bpp destination formats are not supported.
     */
    if ( bit_depth < GRL_BMPDEC_PNG_BIT_DEPTH8 )
    {
        if ( color_type == PNG_COLOR_TYPE_GRAY )
        {
            png_set_expand( png_ptr );
        }
        else
        {
            png_set_packing( png_ptr );
        }
    }

    /* Some settings depending on the colormode */
    switch( color_type )
    {
        /* RGB color data   */
        case PNG_COLOR_TYPE_RGB:
        case PNG_COLOR_TYPE_RGB_ALPHA:
        {
            /* nothing to do */
            break;
        }

        /* Palette data     */
        case PNG_COLOR_TYPE_PALETTE:
        {
            /* Converting palette to RGB    */
            png_set_palette_to_rgb( png_ptr );
            break;
        }

        /* Gray scale data  */
        case PNG_COLOR_TYPE_GRAY:
        case PNG_COLOR_TYPE_GRAY_ALPHA:
        {
            /* Converting gray scale to RGB */
            png_set_gray_to_rgb( png_ptr );
            break;
        }

        default:
        {
            break;
        }
    }
}


GRL_error GRL_BMPDEC_draw_image_PNG ( GRL_BMPDEC_dec_info *p_decode_info )
{
    SVGError            ret_err         = SVG_NO_ERROR; /* Return code                          */
    SVGError            dec_err         = SVG_NO_ERROR; /* Internal error code                  */
    GRL_BMPDEC_png_dec_conf dec_conf    = {NULL, 0, 0, 0, 0, 0, NULL, {0, 0, 0, 0}};          /* Settings for decoding                */
    SVGUint32           cnt_row         = 0;            /* Row counter                          */
    SVGUint8*           dst_color_adrs_org = NULL;      /* Draw address                         */
    SVGUint8*           tmp_color_adrs_org = NULL;      /* Temp. draw buffer address            */
    SVGUint8*           p_read_data     = NULL;         /* Pointer to decoded row               */        
    SVGImage            *p_image        = NULL;         /* Pointer to Image str                 */
    SVGUint32           inc_byte_src    = 0;            /* Increment byte for src               */
    png_structp         png_ptr         = NULL;         /* Handler for pnglib                   */
    png_infop           info_ptr        = NULL;         /* Handler for pnglib                   */
    png_uint_32         width           = 0;            /* PNG image width                      */
    png_uint_32         height          = 0;            /* PNG image height                     */
    SVGInt32            bit_depth       = 0;            /* Bit depth for element                */
    SVGInt32            color_type      = 0;            /* Color type for PNG                   */
    SVGInt32            interlace_type  = 0;            /* Interlace type                       */
    png_bytep           trans           = NULL;         /* Transparency data                    */
    SVGInt32            num_trans       = 0;            /* Num of transparency                  */
    png_color_16p       trans_value     = 0;            /* Colorkey for non-palette images      */

    GRL_BMPDEC_plte_data plte_data      = {0, 0, 0, {0}, {0}, 0, 0, {0}, {0}, {0}, 0, NULL, NULL, NULL};          /* Palette data                         */
    png_bytep           *p_dst_rows     = NULL;         /* Pointer to array of row pointers     */
    SVGBoolean          is_clipped      = SVG_FALSE;    /* Image has to be clipped              */

    SVG_BMP_U("ENTER INTO  GRL_BMPDEC_DRAW_IMAGE_PNG");

    /*
     * Setup bitmap decoding
     */
    if ( NULL == p_decode_info )
    {
        ret_err = SVG_POINTER_NULL;
        SVG_BMP_E("SVG_POINTER_NULL IN GRL_BMPDEC_DRAW_IMAGE_PNG");
    }

    if ( SVG_NO_ERROR == ret_err )
    {
/* PRQA: Lint Message 613: Pointer p_decode_info was checked but Lint does not understand the condition with ret_err */
/*lint -save -e613 */
/* PRQA: QAC Message 505: Pointer p_decode_info was checked but QAC does not understand the condition with ret_err */
        p_image = &(p_decode_info->image);
/*lint -restore */
        
    }

    if ( SVG_NO_ERROR == ret_err )
    {
        plte_data.use_external_palette = SVG_FALSE;
        plte_data.external_TRNS = NULL;
        plte_data.external_PLTE = NULL;

/* PRQA: Lint Message 613: Pointer p_image was checked but Lint does not understand the condition with ret_err */
/*lint -save -e613 */
/* PRQA: QAC Message 505: Pointer p_image was checked but QAC does not understand the condition with ret_err */
        /* Handle the palette replacement */
        if(p_image->type  == SVG_BMPDEC_PNG_ENCODING_WITH_PLTE)
        {
            ret_err = GRL_BMPDEC_palette_replace( p_image, &plte_data );
        }
    }

    if ( SVG_NO_ERROR == ret_err )
    {
        ret_err = GRL_BMPDEC_png_dest_set( p_image, &dec_conf );
    }

    if ( SVG_NO_ERROR == ret_err )
    {
        /* Setup PNG decoding */
        png_ptr = png_create_read_struct( PNG_LIBPNG_VER_STRING,
                                          NULL,
                                          NULL,
                                          NULL );
        if ( png_ptr == NULL )
        {
            ret_err = SVG_BMPDEC_DEC_ERROR;
            SVG_BMP_E("SVG_BMPDEC_DEC_ERROR IN GRL_BMPDEC_DRAW_IMAGE_PNG");
        }
    }

    if ( SVG_NO_ERROR == ret_err )
    {
        info_ptr = png_create_info_struct( png_ptr );
        if ( info_ptr == NULL )
        {
            png_destroy_read_struct( &png_ptr, (png_infopp)NULL , (png_infopp)NULL );
            ret_err = SVG_BMPDEC_DEC_ERROR;
            SVG_BMP_E("SVG_BMPDEC_DEC_ERROR IN GRL_BMPDEC_DRAW_IMAGE_PNG");
        }
    }

    if ( SVG_NO_ERROR == ret_err )
    {
        /*
         * Set callback point for error handling.
         * setjmp/longjmp is not nice but necessary here because the pnglib requires that
         * the error handling function does not return. The alternative would
         * be to abort/crash the module which is even less nice. 
         */
#if PNG_LIBPNG_VER_MAJOR == 1 && PNG_LIBPNG_VER_MINOR >= 6
        if ( 0 != setjmp(png_ptr->jmp_buf_local) )
#else
        if ( 0 != setjmp(png_ptr->jmpbuf) )
#endif
        {
            /* Free PNG structures */
            png_destroy_read_struct( &png_ptr, &info_ptr, (png_infopp)NULL );

            ret_err = SVG_BMPDEC_DEC_ERROR;
            SVG_BMP_E("SVG_BMPDEC_DEC_ERROR IN GRL_BMPDEC_DRAW_IMAGE_PNG");
        }
    }

    if ( SVG_NO_ERROR == ret_err )
    {
        /* Set read function for image data */    
        if ( (p_image->type == SVG_BMPDEC_PNG_ENCODING_WITH_PLTE) &&
             (SVG_TRUE == plte_data.use_external_palette))
        {
            /* Use special read function to inject external palette */
            plte_data.io_ptr = p_image->datap;

            png_set_read_fn( png_ptr,
                             (void*)&plte_data,
                             (png_rw_ptr)GRL_BMPDEC_png_read_func_advanced );
        }
        else
        {
            /* Use normal read function for decoding without external palette */
            png_set_read_fn( png_ptr,
                             p_image->datap,
                             (png_rw_ptr)GRL_BMPDEC_png_read_func );
        }

        /* Read png information */
        png_read_info( png_ptr, info_ptr );
        png_get_IHDR( png_ptr,
                      info_ptr,
                      &width,
                      &height,
                      &bit_depth,
                      &color_type,
                      &interlace_type,
                      NULL,
                      NULL );

        
        /* Clipping setting */
        if ( (dec_conf.src_clip.x == 0) && (dec_conf.src_clip.y == 0) &&
             (dec_conf.src_clip.width == 0) && (dec_conf.src_clip.height == 0) )
        {
            dec_conf.src_clip.width  = width;
            dec_conf.src_clip.height = height;
        }
        dec_err = GRL_BMPDEC_clipping_setup( (SVGInt32)dec_conf.dst_width,
                                             (SVGInt32)dec_conf.dst_height,
                                             &dec_conf.dst_xpos,
                                             &dec_conf.dst_ypos,
                                             &dec_conf.src_clip,
                                             (SVGUint16)width,
                                             (SVGUint16)height,
                                             &is_clipped );
        if ( dec_err != GRL_NO_ERROR )
        {
            ret_err = dec_err;
        }
    }

    if ( SVG_NO_ERROR == ret_err )
    {
        
        /* Deactivate palette replacement if bitmap has no palette */
        if ( color_type != PNG_COLOR_TYPE_PALETTE )
        {
            plte_data.use_external_palette      = SVG_FALSE;
            plte_data.external_TRNS             = NULL;
            plte_data.external_PLTE             = NULL;
        }

        GRL_BMPDEC_png_color_set( png_ptr, bit_depth, color_type );

        /* Read transparency data */
        png_get_tRNS( png_ptr,
                      info_ptr,
                      &trans,
                      &num_trans,
                      &trans_value );
        png_get_valid( png_ptr,
                       info_ptr,
                       PNG_INFO_tRNS );

        /* Let the pnglib merge any tRNS transparency info into the alphachannel */
        png_set_tRNS_to_alpha( png_ptr );

        /* Set filler, i.e. if the image has no alphachannel, create one with full opacity */
        png_set_filler( png_ptr, GRL_BMPDEC_PNG_ALPHA_FILLER, PNG_FILLER_AFTER );

        /* Convert RGBA to BGRA */
        if(p_decode_info->revert == SVG_TRUE)
        {
        	png_set_bgr( png_ptr );
        }
        
        /* Update info struct and get the number of channels for setting the correct read pointer increment */
        png_read_update_info( png_ptr, info_ptr );
        inc_byte_src = (SVGUint32)png_get_channels( png_ptr, info_ptr );


        if ( SVG_TRUE == is_clipped )
        {
            if ( PNG_INTERLACE_ADAM7 == interlace_type )
            {
                /* 
                 * If the bitmap needs to be clipped and it is interlaced,
                 * we need to temporarily allocate a full buffer for the 
                 * decoded bitmap. This is because the normal line-by-line
                 * approach which is used when clipping does not work for
                 * interlaced images where the lines are decoded out-of-order
                 */

				tmp_color_adrs_org = (U8*) malloc(height * width * inc_byte_src);

				if ( tmp_color_adrs_org != NULL ) {

					/* Allocate memory for the row pointer array */

					p_dst_rows = (png_bytep*) malloc( height * sizeof(png_bytep) );
					if ( p_dst_rows != NULL ) {

                        for ( cnt_row = 0; cnt_row < height; cnt_row++ )
                        {
                            p_dst_rows[cnt_row] = (png_bytep)(tmp_color_adrs_org + (cnt_row * width * inc_byte_src));
                        }

                        /* Decode image to temp buffer */
                        png_read_image( png_ptr, p_dst_rows );

/* PRQA: Lint Message 613: Availability of surface is checked in GRL_BMPDEC_check_mode_svg and p_dst_mem is based on that */
/*lint -save -e613 */
/* PRQA: QAC Message 505: Availability of surface is checked in GRL_BMPDEC_check_mode_svg and p_dst_mem is based on that */
                        /* Compute start address in destination memory */
                        dst_color_adrs_org = (U8*)&dec_conf.p_dst_mem[((SVGUint32)dec_conf.dst_xpos * inc_byte_src) 
                                                                      + ((SVGUint32)(dec_conf.dst_ypos ) * dec_conf.dst_mwidth)];
/*lint -restore */

                        /* Now copy the not clipped part to the destination */
                        p_read_data = &tmp_color_adrs_org[((dec_conf.src_clip.y * width * inc_byte_src) 
                                                           + (SVGUint32)(dec_conf.src_clip.x * inc_byte_src))];
                        for ( cnt_row = 0; cnt_row < dec_conf.src_clip.height; cnt_row++ )
                        {
                        	/* Copy line to destination */
                            memcpy( dst_color_adrs_org, p_read_data, (SVGUint32)(dec_conf.src_clip.width * inc_byte_src) );
            
                            dst_color_adrs_org = &dst_color_adrs_org[dec_conf.dst_mwidth];
                            p_read_data        = &p_read_data[(SVGUint32)(width * inc_byte_src)];
                        }

                        free( tmp_color_adrs_org );
                        free( p_dst_rows );
                    }
                    else
                    {
						free( tmp_color_adrs_org );
                    }
                }
            }
            else /* ( PNG_INTERLACE_ADAM7 == interlace_type ) */
            {
                /*
                 * If the bitmap needs to be clipped and is not interlaced,
                 * we can just decode the image line by line and handle
                 * the clipping on the fly. A temporary buffer just for
                 * one line is sufficient.
                 */

            	p_read_data = (png_bytep) malloc( width*inc_byte_src );

            	if ( p_read_data != NULL )
                {
                    /* Clipping top of image by decoding the lines to nowhere */
                    for ( cnt_row = 0; cnt_row < dec_conf.src_clip.y; cnt_row++ )
                    {
                        png_read_row( png_ptr, NULL, NULL );
                    }
/* PRQA: Lint Message 613: Availability of surface is checked in GRL_BMPDEC_check_mode_svg */
/*lint -save -e613 */
                    /* Compute start address in destination memory */
                    dst_color_adrs_org = (U8*)&dec_conf.p_dst_mem[((SVGUint32)dec_conf.dst_xpos * inc_byte_src) 
                                                         + ((SVGUint32)(dec_conf.dst_ypos ) * dec_conf.dst_mwidth)];
/*lint -restore */
                    for ( cnt_row = 0; cnt_row < dec_conf.src_clip.height; cnt_row++ )
                    {
                        /* Decode one row */
                        png_read_row( png_ptr, (png_bytep)p_read_data, NULL );

                        /* Copy line to destination */
                        memcpy( dst_color_adrs_org,
                                &p_read_data[(dec_conf.src_clip.x * inc_byte_src)],
                                (SVGUint32)(dec_conf.src_clip.width * inc_byte_src) );
            
                        dst_color_adrs_org = &dst_color_adrs_org[dec_conf.dst_mwidth];
                    }
					free(p_read_data);
                }
                
            }
        }
        else /* ( SVG_TRUE == is_clipped )*/
        {
            /* Allocate memory for the row pointer array */
        	p_dst_rows = (png_bytep*) malloc( height * sizeof(png_bytep) );

        	if ( p_dst_rows != NULL )
            {       
/* PRQA: Lint Message 613: Availability of surface is checked in GRL_BMPDEC_check_mode_svg */
/*lint -save -e613 */
                /* Compute start address in destination memory */
                dst_color_adrs_org = (U8*)&dec_conf.p_dst_mem[((SVGUint32)dec_conf.dst_xpos * inc_byte_src) 
                                                     + ((SVGUint32)dec_conf.dst_ypos * dec_conf.dst_mwidth)];
/*lint -restore */

                /* Construct array of pointers to the rows */
                for ( cnt_row = 0; cnt_row < dec_conf.src_clip.height; cnt_row++ )
                {
                    p_dst_rows[cnt_row] = (png_bytep)(dst_color_adrs_org + (cnt_row * dec_conf.dst_mwidth));
                }

                /* Decode image */
                png_read_image( png_ptr, p_dst_rows );

            }
            free(p_dst_rows);
            }

        /* Close PNG decoding */
        png_destroy_read_struct( &png_ptr, &info_ptr, (png_infopp)NULL );
    }

    SVG_BMP_U("EXIT FROM GRL_BMPDEC_DRAW_IMAGE_PNG");

    return      ret_err;
}


SVGError GRL_BMPDEC_get_image_info_PNG ( const SVGContextBmpDec  *ctx,
                                         const SVGImage          *p_image,
                                         SVGImageInfo            *image_info )
{
    volatile SVGError       ret_err         = GRL_NO_ERROR;
    png_structp             png_ptr = NULL;
    png_infop               info_ptr= NULL;
    png_uint_32             width;                      /* PNG image width              */
    png_uint_32             height;                     /* PNG image height             */
    S32                     bit_depth;                  /* Bit depth for color */
    S32                     color_type;                 /* Color type for PNG   */
    S32                     interlace_type;             /* Interlaced or not interlaced */
    S32                     compression_type;           /* PNG compression type */
    S32                     filter_method;              /* PNG filter type */
    SVGUint32               chunk_valid     = 0;        /* Chunk info which was read is valid */
    png_bytep               trans           = NULL;     /* Transparency data    */
    SVGInt32                num_trans       = 0;        /* Num of transparency  */
    png_color_16p           trans_value     = NULL;     /* Colorkey for non-palette images  */
    png_color_16p           background      = NULL;     /* PNG background color */
    png_timep               mod_time;                   /* PNG modification time */

    SVG_BMP_U("ENTER INTO GRL_BMPDEC_GET_IMAGE_INFO_PNG");

    if ( (ctx == NULL) || (p_image == NULL) || (image_info == NULL) )
    {
        ret_err = SVG_POINTER_NULL;
        SVG_BMP_E("SVG_POINTER_NULL IN GRL_BMPDEC_GET_IMAGE_INFO_PNG");
    }
    else
    {
        /* Initialize image info */
        memset( image_info, 0x00, sizeof(SVGImageInfo) );

        /*
         * Setup PNG decoding
         */
        png_ptr = png_create_read_struct( PNG_LIBPNG_VER_STRING, NULL, NULL, NULL );
        if ( png_ptr == NULL )
        {
            ret_err = SVG_BMPDEC_DEC_ERROR;
            SVG_BMP_E("SVG_BMPDEC_DEC_ERROR IN GRL_BMPDEC_GET_IMAGE_INFO_PNG");
        }
        else
        {

            info_ptr = png_create_info_struct( png_ptr );
            if ( info_ptr == NULL )
            {
                png_destroy_read_struct( &png_ptr, (png_infopp)NULL , (png_infopp)NULL );
                ret_err = SVG_BMPDEC_DEC_ERROR;
                SVG_BMP_E("SVG_BMPDEC_DEC_ERROR IN GRL_BMPDEC_GET_IMAGE_INFO_PNG");
            }
            else
            {
                /*
                 * Set callback point for error handling.
                 * setjmp/longjmp is not nice but necessary here because the pnglib requires that
                 * the error handling function does not return. The alternative would
                 * be to abort/crash the module which is even less nice. 
                 */
/* PRQA: QAC Message 5122: The libpng uses longjmp so we have to use setjmp here to capture errors */
#if PNG_LIBPNG_VER_MAJOR == 1 && PNG_LIBPNG_VER_MINOR >= 6
                if ( 0 != setjmp(png_ptr->jmp_buf_local) )
#else
                if ( 0 != setjmp(png_ptr->jmpbuf) )
#endif
                {
                    
                    png_destroy_read_struct( &png_ptr, &info_ptr, (png_infopp)NULL );
                    ret_err = SVG_BMPDEC_DEC_ERROR;
                    SVG_BMP_E("SVG_BMPDEC_DEC_ERROR IN GRL_BMPDEC_GET_IMAGE_INFO_PNG");
                    return ret_err;                  /* ======== leave function ======== */
                }

                /* Set read function for image data */
                png_set_read_fn( png_ptr, p_image->datap, (png_rw_ptr)GRL_BMPDEC_png_read_func );

                /* Read png information */
                png_read_info( png_ptr, info_ptr );
                png_get_IHDR( png_ptr,
                              info_ptr,
                              &width,
                              &height,
                              &bit_depth,
                              &color_type,
                              &interlace_type,
                              &compression_type,
                              &filter_method );

                /* Fill image info structure with PNG info */
                image_info->type        = p_image->type;
                image_info->width       = (SVGUint16)width;
                image_info->height      = (SVGUint16)height;
                image_info->bit_depth   = (SVGInt32)bit_depth;

                /* Fill color mode */
                switch ( color_type )
                {
                    case PNG_COLOR_TYPE_GRAY:
                    {
                        image_info->format.png.color_mode = SVG_BMPDEC_PNG_COLOR_TYPE_GRAY;
                        break;
                    }
                    case PNG_COLOR_TYPE_GRAY_ALPHA:
                    {
                        image_info->format.png.color_mode = SVG_BMPDEC_PNG_COLOR_TYPE_GRAY_ALPHA;
                        break;
                    }
                    case PNG_COLOR_TYPE_PALETTE:
                    {
                        image_info->format.png.color_mode = SVG_BMPDEC_PNG_COLOR_TYPE_PALETTE;
                        break;
                    }
                    case PNG_COLOR_TYPE_RGB:
                    {
                        image_info->format.png.color_mode = SVG_BMPDEC_PNG_COLOR_TYPE_RGB;
                        break;
                    }
                    case PNG_COLOR_TYPE_RGB_ALPHA:
                    {
                        image_info->format.png.color_mode = SVG_BMPDEC_PNG_COLOR_TYPE_RGB_ALPHA;
                        break;
                    }
                    default:
                    {
                        image_info->format.png.color_mode = SVG_BMPDEC_PNG_COLOR_TYPE_UNKNOWN;
                    }
                }

                /* Fill alpha channel info */
                chunk_valid = png_get_tRNS( png_ptr, info_ptr, &trans, &num_trans, &trans_value );
                if ( (((SVGUint32)color_type & PNG_COLOR_MASK_ALPHA) != 0) ||
                     ( (((SVGUint32)color_type & PNG_COLOR_MASK_PALETTE) != 0)
                       && (0 != chunk_valid) && (0 < num_trans)) )       /* For paletted PNG we need to check tRNS, too */
                {
                    image_info->format.png.has_alphachannel = SVG_TRUE;
                }
                else
                {
                    image_info->format.png.has_alphachannel = SVG_FALSE;
                }

                /* Fill interlaced info */
                if ( PNG_INTERLACE_ADAM7 == interlace_type )
                {
                    image_info->format.png.is_interlaced = SVG_TRUE;
                }
                else
                {
                    image_info->format.png.is_interlaced = SVG_FALSE;
                }

                /* Fill filter method */
                if ( PNG_FILTER_TYPE_BASE == filter_method )
                {
                    image_info->format.png.filter_method = SVG_BMPDEC_PNG_FILTER_METHOD_BASE;
                }
                else
                {
                    /* Unsupported filter type */
                    SVG_BMP_W("SVG_NO_ERROR IN GRL_BMPDEC_GET_IMAGE_INFO_PNG");
                    image_info->format.png.filter_method = SVG_BMPDEC_PNG_FILTER_METHOD_UNKNOWN;
                    
                }

                /* Fill background color info */
                chunk_valid = png_get_bKGD( png_ptr,
                                            info_ptr,
                                            &background );
                if ( 0 != chunk_valid )
                {
                    image_info->format.png.has_bg_color = SVG_TRUE;
                    /* Convert to RGB888 */
                    image_info->format.png.bg_color = ((((SVGUint32)background->red   >> GRL_BMPDEC_PNG_BIT_DEPTH8) 
                                                        << GRL_BMPDEC_PNG_BIT_DEPTH16) |
                                                       (((SVGUint32)background->green >> GRL_BMPDEC_PNG_BIT_DEPTH8) 
                                                        << GRL_BMPDEC_PNG_BIT_DEPTH8)  |
                                                       ((SVGUint32)background->blue  >> GRL_BMPDEC_PNG_BIT_DEPTH8) );
                }
                else
                {
                    image_info->format.png.has_bg_color = SVG_FALSE;
                    image_info->format.png.bg_color     = 0;
                }

                /* Fill modification time info */
                chunk_valid = png_get_tIME( png_ptr,
                                            info_ptr,
                                            &mod_time );
                if ( 0 != chunk_valid )
                {
                    image_info->format.png.has_mod_time         = SVG_TRUE;
                    image_info->format.png.mod_time.year        = mod_time->year;
                    image_info->format.png.mod_time.month       = mod_time->month;
                    image_info->format.png.mod_time.day         = mod_time->day;
                    image_info->format.png.mod_time.hour        = mod_time->hour;
                    image_info->format.png.mod_time.minute      = mod_time->minute;
                    image_info->format.png.mod_time.second      = mod_time->second;
                }
                else
                {
                    image_info->format.png.has_mod_time = SVG_FALSE;
                    image_info->format.png.mod_time.year        = 0;
                    image_info->format.png.mod_time.month       = 0;
                    image_info->format.png.mod_time.day         = 0;
                    image_info->format.png.mod_time.hour        = 0;
                    image_info->format.png.mod_time.minute      = 0;
                    image_info->format.png.mod_time.second      = 0;
                }

                /* Free the allocated resources */
                png_destroy_read_struct( &png_ptr, &info_ptr, (png_infopp)NULL );
            }
        }
    }
    
    SVG_BMP_U("EXIT FROM GRL_BMPDEC_GET_IMAGE_INFO_PNG");

    return  ret_err;
}

